<?php

namespace pxlrbt\FilamentChangelog;

use Closure;
use Filament\Facades\Filament;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use pxlrbt\FilamentChangelog\Actions\DownloadAssetsAction;
use pxlrbt\FilamentChangelog\Connectors\GithubConnector;
use pxlrbt\FilamentChangelog\DataObjects\Release;

class Changelog
{
    private static ?Closure $filterReleasesUsing = null;

    public static function getClient(): GithubConnector
    {
        return app(GithubConnector::class);
    }

    public static function releases(): Collection
    {
        return cache()->rememberForever(config('filament-changelog.cache_key'), function () {
            $releases = collect();
            $client = static::getClient();

            $response = $client->get("repos/{$client->getRepo()}/releases", ['per_page' => 100]);

            if ($response->successful()) {
                $releases = $releases->concat($response->collect()->all());
            }

            return $releases
                ->map(fn (array $release) => Release::fromGithub($release))
                ->filter(function (Release $release) {
                    if (self::$filterReleasesUsing === null) {
                        return true;
                    }

                    return (self::$filterReleasesUsing)($release);
                })
                ->each(fn (Release $release) => (new DownloadAssetsAction)($release));
        });
    }

    public static function latestTag(): ?string
    {
        return self::releases()->first()?->tag;
    }

    public static function clear()
    {
        cache()->forget(config('filament-changelog.cache_key'));

        Storage::disk(config('filament-changelog.storage.disk'))
            ->deleteDirectory(config('filament-changelog.storage.path'));
    }

    public static function filterReleasesUsing(?Closure $callback)
    {
        self::$filterReleasesUsing = $callback;
    }

    public static function lastSeenVersion(): ?string
    {
        $plugin = Filament::getPlugin('pxlrbt/filament-changelog');

        if (($column = $plugin->userVersionColumn) === null) {
            return null;
        }

        return Filament::auth()->user()->{$column};
    }

    public static function hasUserSeenLatest(): bool
    {
        return version_compare(
            self::lastSeenVersion(),
            self::latestTag(),
            '>='
        );
    }

    public static function markAsViewed(): void
    {
        $plugin = Filament::getPlugin('pxlrbt/filament-changelog');

        if (($column = $plugin->userVersionColumn) === null) {
            return;
        }

        Filament::auth()->user()->update([$column => self::latestTag()]);
    }
}
