<?php

namespace pxlrbt\FilamentChangelog\DataObjects;

use Carbon\CarbonImmutable;
use Illuminate\Support\Facades\Storage;
use pxlrbt\FilamentChangelog\Connectors\GithubConnector;

readonly class Release
{
    public function __construct(
        public string $id,
        public string $name,
        public string $tag,
        public string $body,

        public array $assets,

        public string $created_at,
        public string $published_at,

    ) {
    }

    public static function fromGithub(array $data): static
    {
        return new self(
            id: $data['id'],
            name: $data['name'],
            tag: $data['tag_name'],
            body: $data['body'],
            assets: $data['assets'],
            created_at: $data['created_at'],
            published_at: $data['published_at'],
        );
    }

    public function publishedAt(): CarbonImmutable
    {
        return CarbonImmutable::parse($this->published_at);
    }

    public function createdAt(): CarbonImmutable
    {
        return CarbonImmutable::parse($this->published_at);
    }

    public function body()
    {
        $body = strtr($this->body, ['###' => '####', '##' => '###']);
        $body = $this->replaceAssetUrls($body);

        return str()->markdown($body);
    }

    public function replaceAssetUrls(string $body): string
    {
        $client = app(GithubConnector::class);

        $assetBaseUrl = "https://github.com/{$client->getRepo()}/assets";
        $disk = config('filament-changelog.storage.disk');
        $path = config('filament-changelog.storage.path');

        return str_replace(
            $assetBaseUrl,
            Storage::disk($disk)->url($path.'/'),
            $body
        );
    }
}
